#! /usr/bin/env perl

use strict;
use warnings;
use Pod::Usage;
use Smash::Global qw($SMASH_SCRIPT_NAME);
use Smash::CommandLineParser qw(parse_options check_required_options);
use Smash::Core;

##############
# Set up command line parsing
##############

my @allowed  = qw(collection=s title=s description=s help); # arguments I expect
my @required = qw(collection title description);   # arguments I require

##############
# Parse command line options
##############

my $status;
my $missing;
my %options;

($status, %options) = parse_options(\@allowed);
if ($options{help}) {
	pod2usage(-exitstatus => 0, -verbose => 2);
}
if ($status != 1) {
	pod2usage(-message => "", -exitstatus => 2, -verbose => 1);
}
#print_options(%options);
($status, $missing) = check_required_options(\@required, %options);
if ($status != 1) {
	pod2usage(-message => "$SMASH_SCRIPT_NAME: Missing argument --$missing\n", -exitstatus => 2, -verbose => 1);
}

##############
# Add
##############

my $collection = $options{collection};
my $smash      = new Smash::Core(COLLECTION => $collection);
$smash->init();
my $metagenome = $smash->make_new_metagenome($collection, $options{title}, $options{description});
$smash->finish();

print "<output>$metagenome</output>\n";
print "********************************************************\n";
print "  Metagenome $metagenome added to $collection\n";
print "********************************************************\n";

exit(0);

=head1 Name

addMetaGenome.pl - Wrapper script to add a metagenome to a collection in Smash

=head1 Synopsis

	addMetaGenome.pl [options]

=head1 Options

=over 4

=item B<C<--collection>> (required)

Name of metagenome collection to which this metagenome is added

=item B<C<--title>>

A short title to identity this metagenome. We recommend using something short,
like under 8 characters, since this will be used in analysis results and
displays. Long names will make your figures look ugly!

=item B<C<--description>>

A brief description of this metagenome.

=item B<C<--help>>

Prints this manual.

=back

=head1 Description

B<addMetaGenome.pl> is a wrapper script to add a metagenome to a collection in Smash.

A typical use of B<addMetaGenome.pl> is like this:

	addMetaGenome.pl --collection=MC20 --title=AMD \
		--description="Acid Mine Drainage data from Tyson et al, 2004"

Here the title B<AMD> is short, but enough to identify the famous AMD dataset. If you
are working with human gut metagenomes like we are, you could use 

	addMetaGenome.pl --collection=MC20 --title=JP-AD-1 \
		--description="Human gut metagenome from Japanese adult F1-S"

=cut
