#! /usr/bin/env perl

use strict;
use warnings;
use Pod::Usage;
use Smash::Global qw($SMASH_SCRIPT_NAME);
use Smash::CommandLineParser qw(parse_options check_required_options);
use Smash::Core;

##############
# Set up command line parsing
##############

my @allowed  = qw(title=s description=s collection=s remove unload wipeout help); # arguments I expect
my @required = ();   # arguments I require

##############
# Parse command line options
##############

my $status;
my $missing;
my %options;

($status, %options) = parse_options(\@allowed);
if ($options{help}) {
	pod2usage(-exitstatus => 0, -verbose => 2);
}
if ($status != 1) {
	pod2usage(-message => "", -exitstatus => 2, -verbose => 1);
}
#print_options(%options);

##############
# Add
##############

my $smash      = new Smash::Core();
$smash->init();
if ($options{unload} || $options{wipeout} || $options{remove}) {
	@required = qw(collection);
	($status, $missing) = check_required_options(\@required, %options);
	if ($status != 1) {
		pod2usage(-message => "$SMASH_SCRIPT_NAME: Missing argument --$missing\n", -exitstatus => 2, -verbose => 1);
	}
	my $collection = $options{collection};
	$smash->remove_collection($collection);
	print "<output>success</output>\n";
	print "********************************************************\n";
	print "  Metagenome collection $collection removed from SMASH\n";
	print "********************************************************\n";
} else {
	@required = qw(title description);
	($status, $missing) = check_required_options(\@required, %options);
	if ($status != 1) {
		pod2usage(-message => "$SMASH_SCRIPT_NAME: Missing argument --$missing\n", -exitstatus => 2, -verbose => 1);
	}
	my $new_collection = $smash->make_new_collection($options{title}, $options{description});
	print "<output>$new_collection</output>\n";
	print "********************************************************\n";
	print "  Metagenome collection $new_collection added to SMASH\n";
	print "********************************************************\n";
}
$smash->finish();

exit(0);

=head1 Name

addMetaGenomeCollection.pl - Wrapper script to add a metagenome collection to Smash

=head1 Synopsis

	addMetaGenomeCollection.pl [options]

=head1 Options

=over 4

=item B<C<--title>> (required)

A short title to identity this collection. We recommend using something short,
like under 8 characters, since this will be used in analysis results and
displays. Long names will make your figures look ugly!

=item B<C<--description>> (required)

A brief description of this metagenome.

=item B<C<--remove>>, B<C<--unload>>, B<C<--wipeout>>

Remove this collection from Smash (requires C<--collection>)

=item B<C<--collection>>

collection to be removed.

=item B<C<--help>>

Prints this manual.

=back

=head1 Description

B<addMetaGenome.pl> is a wrapper script to add a metagenome to a collection in Smash.

A typical use of B<addMetaGenome.pl> is like this:

	addMetaGenomeCollection.pl --title=Pub2008 \
		--description="Collection of metagenomes published before 2008"

If you are working with human gut metagenomes like we are, you could use 

	addMetaGenomeCollection.pl --title=Human_Gut \
		--description="Collection of human gut metagenomes from US, Japan, France and Italy"

=cut
