#! /usr/bin/env perl

use strict;
use warnings;
use Pod::Usage;
use Smash::Global qw($SMASH_SCRIPT_NAME);
use Smash::CommandLineParser qw(parse_options check_required_options);
use Smash::Databases::MetaGenomeDB::Loader qw(load_smash);

##############
# Set up command line parsing
##############

my @allowed  = qw(assembly=s unload wipeout help); # arguments I expect
my @required = qw(assembly);                       # arguments I require

##############
# Parse command line options
##############

my $status;
my $missing;
my %options;

($status, %options) = parse_options(\@allowed);
if ($options{help}) {
	pod2usage(-exitstatus => 0, -verbose => 2);
}
if ($status != 1) {
	pod2usage(-message => "", -exitstatus => 2, -verbose => 1);
}
#print_options(%options);
($status, $missing) = check_required_options(\@required, %options);
if ($status != 1) {
	pod2usage(-message => "$SMASH_SCRIPT_NAME: Missing argument --$missing\n", -exitstatus => 2, -verbose => 1);
}

##############
# Handle command line options
# (except checking for presence of required args, which has already 
# been done by parse_options)
##############

##############
# Load
##############

load_smash("assembly", %options);

print "<output>success</output>\n";
print "********************************************************\n";
if ($options{wipeout}) {
print "  Assembly $options{assembly} wiped out from SMASH\n";
} elsif ($options{unload}) {
print "  Assembly $options{assembly} unloaded from SMASH\n";
} else {
print "  Assembly $options{assembly} loaded to SMASH\n";
}
print "********************************************************\n";

exit(0);

=head1 Name

loadAssembly.pl - Wrapper script to load/unload an assembly to/from Smash database

=head1 Synopsis

	loadAssembly.pl [options]

=head1 Options

=over 4

=item B<C<--assembly>>

Name of assembly to load into Smash database. This assembly should exist in Smash
repository. This script will then load it into the database.

=item B<C<--unload>>

Unloads the entries corresponding to this assembly from the database. This enables 
you to reload the assembly from the files in Smash repository. Use this when you 
suspect something wrong in the database, but you think the assembly itself is fine. 
In such cases, you B<C<--unload>> the assembly and then load it again using this script.

=item B<C<--wipeout>>

Unloads the entries corresponding to this assembly from the database and removes 
all the files from the Smash repository. Once an assembly is B<C<wipedout>>, it 
disappears from Smash. So please be careful when you use this option.

=item B<C<--help>>

Prints this manual.

=back

=head1 Description

B<loadAssembly.pl> is a wrapper script to load an assembly in Smash repository into 
Smash database. The assembly should already exist in the repository and must have 
been generated by a Smash-compatible script such as B<doAssembly.pl>. 

A typical use of B<loadAssembly.pl> follows B<doAssembly.pl>, like so:

	doAssembly.pl --metagenome=MC20.MG1 --assembler=Arachne \
	    --genome_size=20000 --superiterative
	loadAssembly.pl --assembly=MC20.MG1.AS1

=cut
