#! /usr/bin/env perl

use strict;
use warnings;
use Pod::Usage;
use Smash::Global qw($SMASH_SCRIPT_NAME);
use Smash::CommandLineParser qw(parse_options check_required_options);
use Smash::Databases::MetaGenomeDB::Loader qw(load_smash);

##############
# Set up command line parsing
##############

my @allowed  = qw(genepred=s label=s unload wipeout help); # arguments I expect
my @required = qw(genepred);                               # arguments I require

##############
# Parse command line options
##############

my $status;
my $missing;
my %options;

($status, %options) = parse_options(\@allowed);
if ($options{help}) {
	pod2usage(-exitstatus => 0, -verbose => 2);
}
if ($status != 1) {
	pod2usage(-message => "", -exitstatus => 2, -verbose => 1);
}
#print_options(%options);
($status, $missing) = check_required_options(\@required, %options);
if ($status != 1) {
	pod2usage(-message => "$SMASH_SCRIPT_NAME: Missing argument --$missing\n", -exitstatus => 2, -verbose => 1);
}

##############
# Handle command line options
# (except checking for presence of required args, which has already 
# been done by parse_options)
##############

##############
# Load
##############

load_smash("GenePrediction", %options);

print "<output>success</output>\n";
print "********************************************************\n";
if ($options{wipeout}) {
print "  Gene prediction $options{genepred} wiped out from SMASH\n";
} elsif ($options{unload}) {
print "  Gene prediction $options{genepred} unloaded from SMASH\n";
} else {
print "  Gene prediction $options{genepred} loaded to SMASH\n";
}
print "********************************************************\n";

exit(0);

=head1 Name

loadGenePrediction.pl - Wrapper script to load/unload a gene prediction to/from Smash database

=head1 Synopsis

	loadGenePrediction.pl [options]

=head1 Options

=over 4

=item B<C<--genepred>>

Id of gene prediction to load into Smash database. This gene prediction should 
exist in Smash repository. This script will then load it into the database.

=item B<C<--label>>

If this gene prediction was generated using a label in addition to the gene prediction id, 
you can specify it through B<C<--label>>. 
B<label> identifies the files part of a run of gene prediction and by default it is equal to the
gene prediction id. If you specify a B<label> while making gene prediction, you must use the same
B<label> to identify the files. 
This option comes in handy when parallelizing gene prediction. 
You can generate a gene prediction by splitting the input file into parts and running them using 
the same gene prediction id, but different labels. If you specify those labels to this script
through B<C<--label>>, then they will be loaded into the specified gene prediction id in the
database. This is what the B<C<--parallelize>> option in L<doGenePrediction.pl> does.
B<WARNING>: If you don't understand this, don't use this option.

=item B<C<--unload>>

Unloads the entries corresponding to this gene prediction from the database. This enables 
you to reload the gene prediction from the files in Smash repository. Use this when you 
suspect something wrong in the database, but you think the gene prediction itself is fine. 
In such cases, you B<C<--unload>> the gene prediction and then load it again using this script.

=item B<C<--wipeout>>

Unloads the entries corresponding to this gene prediction from the database and removes 
all the files from the Smash repository. Once a gene prediction is B<wipedout>, it
disappears from Smash. So please be careful when you use this option.

=item B<C<--help>>

Prints this manual.

=back

=head1 Description

B<loadGenePrediction.pl> is a wrapper script to load an gene prediction in Smash repository into 
Smash database. The gene prediction should already exist in the repository and must have 
been generated by a Smash-compatible script such as B<doGenePrediction.pl>. 

A typical use of B<loadGenePrediction.pl> follows B<doGenePrediction.pl>, like so:

	doGenePrediction.pl --assembly=MC20.MG1.AS1 --predictor=GeneMark
	loadGenePrediction.pl --genepred=MC20.MG1.AS1.GP1

=cut
