#!/bin/bash

##################################################################
#
#   WARNING   WARNING   WARNING   WARNING   WARNING   WARNING
#
# This is not a generic BLAST wrapper script.
# This is a specific script used by the parallelized BLAST
# option of SmashCommunity. Do not expect this to work by
# running it with a database and query. It is more complicated
# than that. This is usually called with a file containing 
# definitions of variables listed later.
#
# For generic BLAST driver script, check runBlast.pl 
# or for parallelized BLAST check runParallelBlast.pl
#
##################################################################

#################################
# Function definitions
#################################

stage_file_in() {
	local local_dir=$1;
	local lock=$2
	local remote_file=$3;
	if [ -z "$lock" ]; then
		echo "Usage: stage_file_in localdir lockid remotefile" 1>&2;
		exit 2;
	fi;

	local file=$(basename $remote_file);
	local local_copy="$local_dir/$file";

	lockfile -r -1 -l 3600 $remote_file.lock.$lock;
	lockfile -r -1 -l 3600 $local_copy.lock; 

	local attempts=0;
	local copy=1;
	while [ $attempts -lt 10 -a $copy -eq 1 ]; do
		rsync -q -ptg $remote_file $local_copy; 
		if [ $? -ne 0 ] ; then  # rsync failed!
			copy=1;
			attempts=$(expr $attempts + 1);
		else
			copy=0;
		fi;
	done
	if [ $attempts -eq 10 ]; then
		echo "Staging in $remote_file to $local_dir/ failed after $attempts attempts! Please copy it yourself." 1>&2;
	fi
	rm -f $local_copy.lock;
	rm -f $remote_file.lock.$lock;
}

stage_multiple_files_in() {

	local local_location=$1;
	local lock=$2;

	if [ -z "$lock" ]; then
		echo "Usage: stage_multiple_files_in localdir lockid remotefile1 [remotefile2 ...]" 1>&2;
		exit 2;
	fi;

	shift; shift; # skip the 2 arguments 

	mkdir -p $local_location;
	for file in $*; do
		#echo "stage_file_in $file $local_location/ $lock;"
		stage_file_in $local_location/ $lock $file;
	done;

}

stage_file_out() {
	local local_file=$1;
	local remote_dir=$2;
	if [ -z "$remote_dir" ]; then
		echo "Usage: stage_file_out local-file remote-dir" 1>&2;
		exit 2;
	fi;

	local file=$(basename $local_file);
	local remote_copy="$remote_dir/$file";

	local attempts=0;
	local copy=1;
	while [ $attempts -lt 10 -a $copy -eq 1 ]; do
		rsync -q -ptg $local_file $remote_copy;
		if [ $? -ne 0 ] ; then  # rsync failed!
			copy=1;
			attempts=$(expr $attempts + 1);
		else
			copy=0;
		fi;
	done
	if [ $attempts -eq 10 ]; then
		echo "Staging out $local_file to $remote_dir/ failed after $attempts attempts! Please copy it yourself by running:" 1>&2;
		echo "rsync -ptg $(hostname):$local_file $remote_dir/" 1>&2;
	else
		rm -f $local_file;
	fi
}


#################################
# Script itself
#################################

def_file=$1;
nfs_lock=$2;
sleep_time=$3;

if [ -z "$sleep_time" ]; then
	echo "Usage: $0 def-file locknum sleep-time" 1>&2;
	exit 2;
fi;

source $def_file;

if [ -z "$remote_db_location" \
  -o -z "$remote_output_location" \
  -o -z "$local_db_location" \
  -o -z "$local_output_location" \
  -o -z "$database" \
  -o -z "$query" \
  -o -z "$extensions" \
  -o -z "$flavor" \
  -o -z "$blast" \
  -o -z "$blast_exe" \
  -o -z "$blast_database_arg" \
  -o -z "$blast_query_arg" \
  -o -z "$blast_parameters" \
  -o -z "$cpu_arg" ]; then
	echo "One of the following variables is not set in $def_file. Please check the values and set the missing ones:" 1>&2;
	cat 1>&2 <<EOS
remote_db_location=$remote_db_location;
remote_output_location=$remote_output_location;
local_db_location=$local_db_location;
local_output_location=$local_output_location;
query=$query;
database=$database;
extensions=$extensions;
flavor=$flavor;
blast=$blast;
blast_exe='$blast_exe';
blast_database_arg='$blast_database_arg';
blast_parameters='$blast_parameters';
cpu_arg='$cpu_arg';
EOS
	exit 2;
fi

# Stage in BLAST database

sleep $sleep_time;
stage_multiple_files_in $local_db_location $nfs_lock $remote_db_location/$database.$extensions;
#echo stage_multiple_files_in $local_db_location $nfs_lock $remote_db_location/$database.$extensions;

# Figure out local output space

# PBS

if [ "$PBS_ENVIRONMENT" = "PBS_BATCH" -a ! -z "$PBS_JOBID" ]; then
	local_output_location="$local_output_location/$PBS_JOBID";
fi

# SGE

if [ "$ENVIRONMENT" = "BATCH" -a ! -z "$JOB_ID" ]; then
	local_output_location="$local_output_location/$JOB_ID.$SGE_TASK_ID";
fi

# Run BLAST

mkdir -p $local_output_location;
echo "# Running command: $pre_processor $blast_exe $blast_database_arg $blast_query_arg $blast_parameters $cpu_arg $post_processor > $local_output_location/$query.$blast";
eval $pre_processor $blast_exe $blast_database_arg $blast_query_arg $blast_parameters $cpu_arg $post_processor > $local_output_location/$query.$blast;

# Check if the file is complete if there is no post processing
# The script only knows how to handle real BLAST reports, and hence the post processor check

if [ -z "$post_processor" ]; then

	# If bit_threshold is given, then filter based on bit_threshold
	# Otherwise, just check for completeness

	script_dir=$(dirname $0);
	if [ ! -z "$bit_threshold" ]; then
		echo "# Running command: perl $script_dir/filterBlastReport.pl --bits=$bit_threshold --check_complete --fasta=$remote_output_location/$query.fa $local_output_location/$query.$blast > $local_output_location/$query.$blast.filtered";
		eval perl $script_dir/filterBlastReport.pl --bits=$bit_threshold --check_complete --fasta=$remote_output_location/$query.fa $local_output_location/$query.$blast > $local_output_location/$query.$blast.filtered;
		stage_file_out $local_output_location/$query.$blast.filtered $remote_output_location/;
	else
		echo "# Running command: perl $script_dir/filterBlastReport.pl --flavor=$flavor --check_complete --nofilter --fasta=$remote_output_location/$query.fa $local_output_location/$query.$blast";
		eval perl $script_dir/filterBlastReport.pl --flavor=$flavor --check_complete --nofilter --fasta=$remote_output_location/$query.fa $local_output_location/$query.$blast;
	fi
fi

# Stage out result file

stage_file_out $local_output_location/$query.$blast $remote_output_location/;
