#include "mBamVector.h"

/* Generic BAM/SAM utilities */

int32_t bam_cigar2alnlen(const bam1_core_t *c, const uint32_t *cigar) {
	uint32_t k;
	int32_t length = 0;
	for (k = 0; k < c->n_cigar; ++k) {
		int op = cigar[k] & BAM_CIGAR_MASK;
		if (!(op == BAM_CHARD_CLIP || op == BAM_CSOFT_CLIP || op == BAM_CREF_SKIP || op == BAM_CPAD)) {
			length += cigar[k] >> BAM_CIGAR_SHIFT;
		}
	}
	return length;
}

void bam_cigar2details(const bam1_core_t *c, const uint32_t *cigar, int32_t *alen, int32_t *qlen, int32_t *qclip) {
	uint32_t k;
	*alen = *qlen = *qclip = 0;
	for (k = 0; k < c->n_cigar; ++k) {
		int op = cigar[k] & BAM_CIGAR_MASK;
		int w  = cigar[k] >> BAM_CIGAR_SHIFT;
		if (op == BAM_CHARD_CLIP || op == BAM_CSOFT_CLIP) {
			*qclip += w;
			*qlen  += w;
		} else if (!(op == BAM_CREF_SKIP || op == BAM_CPAD)) {
			*alen += w;
			if (op == BAM_CMATCH || op == BAM_CINS)
				*qlen += w;
		}
	}
}

/* mBamVector */

void mSortBamVector(mBamVector *vec, int(*compar)(const void *, const void *)) {
	qsort(vec->elem, vec->size, sizeof(bam1_t*), compar);
}

void mInitBamVector(mBamVector *vec, int limit) {
	vec->limit = (limit>0)?limit:1;
	/* Don't want to handle empty vectors. This is faster. Memory doesnt matter */
	vec->elem = (bam1_t**) mMalloc(limit*sizeof(bam1_t*));
	vec->size = 0;
}

void mPushBamVector(mBamVector *vec, bam1_t *item) {
	if (vec->size == vec->limit) {
#ifdef DEBUG
		fprintf(stderr, "Expanding vector to %d\n", vec->limit *2);
#endif
		vec->limit *= 2;
		vec->elem = (bam1_t**) mRealloc(vec->elem, vec->limit*sizeof(bam1_t*));
	}
	vec->elem[vec->size] = item;
	vec->size++;
}

void mFreeBamVector(mBamVector *vec) {
	mFree(vec->elem);
}

void mEmptyBamVector(mBamVector *vec) {
	int i;
	for (i=0; i<vec->size; i++) {
		if (vec->elem[i] != NULL) {
			bam_destroy1(vec->elem[i]);
		}
	}
	vec->size = 0;
}

void mWriteBamVector(samfile_t *stream, mBamVector *bamvector) {
	int i;
	for (i=0; i<bamvector->size; i++) {
		bam1_t *b = bamvector->elem[i];
		samwrite(stream, b);
	}
}

/* mBamPool */

void mInitBamPool(mBamPool *pool, int limit) {
	int i;
	bam1_t **elem;
	pool->limit = (limit>0)?limit:1;
	pool->elem = (bam1_t**) mMalloc(limit*sizeof(bam1_t*));
	pool->origin = 0;
	pool->current = 0;
	elem = pool->elem;
	for (i=0; i<limit; i++)
		elem[i] = bam_init1();
}

void mExpandBamPool(mBamPool *pool) {
	int i;
	int limit = pool->limit;
	bam1_t **elem = pool->elem;
	elem = (bam1_t**) mRealloc(elem, 2*limit*sizeof(bam1_t*));

	/* the following four lines should NOT use pool->limit, this is the old limit */
	pool->current = pool->origin + limit; 
	if (pool->origin > 0) { /* pool is starting in middle */

		/* copy left of origin to beginning of new region */
		/* memcpy(elem+limit, elem, pool->origin*sizeof(bam1_t*)); */
		/* reinitialize bam in the area left of origin */
		for (i=0; i<pool->origin; i++) {
			elem[i+limit] = elem[i];
			elem[i] = bam_init1();
		}
	}

	/* reinitialize bam to the right of current */
	for (i=pool->current; i<2*limit; i++)
		elem[i] = bam_init1();

	pool->elem  = elem;
	pool->limit *= 2;
}

void mReOriginateBamPool(mBamPool *pool) {
	pool->origin = pool->current;
}

bam1_t* mAdvanceBamPool(mBamPool *pool) {
	pool->current = (pool->current+1)%pool->limit;
	if (pool->current == pool->origin) {
		mExpandBamPool(pool);
#ifdef DEBUG
		fprintf(stderr, "BAM pool expanded to %d!\n", pool->limit);
#endif
	}
	return pool->elem[pool->current];
}

void mFreeBamPool(mBamPool *pool) {
	int i;
	for (i=0; i<pool->limit; i++)
		bam_destroy1(pool->elem[i]);
	mFree(pool->elem);
}

void mWriteBamPool(samfile_t *stream, mBamPool *pool) {
	int i;
	bam1_t **elem = pool->elem;
	if (pool->origin <= pool->current) {
		for (i=pool->origin; i<pool->current; i++) 
			samwrite(stream, elem[i]);
	} else {
		for (i=pool->origin; i<pool->limit; i++) 
			samwrite(stream, elem[i]);
		for (i=0; i<pool->current; i++) 
			samwrite(stream, elem[i]);
	}
}

