#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "mQual.h"

#define COLUMNS 17

void mFreeQual(mQual *qual) {
	mFree(qual->def);
	mFree(qual->data);
}

void mFreeQualVector(mVector *vec) {
	int i;
	for (i=0; i<vec->size; i++) {
		mFreeQual((mQual*)vec->elem[i]);
		mFree(vec->elem[i]);
	}
	mFreeVector(vec);
}

void mInitQual(mQual *qual, int bases) {
	qual->data   = (int*) mCalloc(bases, sizeof(int));
	qual->length = bases;
}

/* returns END_OF_QUAL - if there is no more fasta entry left,
 *         HAS_MORE_QUAL - if there is more fasta entries left
 */
int mReadQual(FILE *stream, mQual *qual) {
	int     next = END_OF_QUAL;
	int    *data;
	char   *def;
	int     c;
	coor_t  count = 0;
	coor_t  seq_limit = 512; /* initial mem alloc */
	coor_t  hdr_limit = 512; /* initial mem alloc */

	data = (int*) mCalloc(seq_limit, sizeof(int));
	def  = (char*) mCalloc(hdr_limit, sizeof(char));

	while (isspace(c = fgetc(stream)));
	if ('>' == (char) c) { /* Header line found */
		while ( (c = fgetc(stream)) != EOF && c != '\n') { 
			/* Process header */
			def[count++] = (char) c;
			if (count == hdr_limit) {
				hdr_limit *= 2;
				def = (char*) mRealloc(def, hdr_limit*sizeof(char));
			}
		}
		def[count] = '\0';
	} else {
		/*mExit("Not a valid Fasta file!");*/
		mDie("Not a valid Fasta file!");
	}

	count = 0;
	for(;;){
		int val;
		if (fscanf(stream, "%d", &val) != 1) {
			c = fgetc(stream);
			if (c == EOF || c == '>') {
				/* next record */
				break;
			}
		}
		data[count] = val;
		if (count == seq_limit) {
			seq_limit *= 2;
			data = (int*) mRealloc(data, seq_limit*sizeof(int));
		}
		count++;
	}
	if (c == '>') {
		/* next record */
		ungetc(c, stream);
		next = HAS_MORE_QUAL;
	}

	qual->length = count;
	qual->def = def;
	qual->data = data;
	return next;
}

int mReadMultipleQual(FILE *stream, mVector *multi) {
	int status;
	mQual*  qual   = (mQual*)  mMalloc(sizeof(mQual));
	while((status = mReadQual(stream, qual))) {
		mPushVector(multi, qual);
		if (status == END_OF_QUAL) { /* Last entry */
			break;
		}
		qual = (mQual*) mMalloc(sizeof(mQual));
	}
	return multi->size;
}

void mWriteMultipleQual(FILE *stream, mVector *multi) {
	int i;
	for (i=0; i<multi->size; i++) {
		mWriteQual(stream, (mQual*)multi->elem[i]);
	}
}

void mWriteQual(FILE *stream, mQual *qual) {
	int i;
	int   length = qual->length;
	int  *data   = qual->data;
	fprintf(stream, ">%s\n", qual->def);
	for (i=0; i < length; i++) {
		fprintf(stream, "%d", data[i]);
		if (i%COLUMNS == COLUMNS-1) {
			fprintf(stream, "\n");
		} else {
			fprintf(stream, " ");
		}
	}
	if (length%COLUMNS != 0) fprintf(stream, "\n");
}

void mReverseQual(mQual *qual) {
	int  i;
	int  length = qual->length;
	int *data   = qual->data;
	for (i=0; i<length/2; i++) {
		int tmp = data[length-i-1];
		data[length-i-1] = data[i];
		data[i] = tmp;
	}
}

void mReverseComplementQual(mQual *qual) {
	mReverseQual(qual);
}

void mProcessQualDef(mQual *qual) {
	char *def    = qual->def;
	int   length = strlen(def);
	int   i;
	for (i=0; i<length; i++) {
		if (isspace(def[i])) {
			def[i] = '\0';
		}
	}
}

mQual* mSubQual(mQual *qual, coor_t offset, coor_t length) {
	mQual *sub;

	if (offset >= qual->length) return NULL;

	sub         = (mQual*) mMalloc(sizeof(mQual));
	sub->data   = qual->data + (int)offset;
	sub->def    = qual->def;
	sub->length = length;
	if (offset + length > qual->length) {
		sub->length = qual->length - offset;
	}
	return sub;
}

char* mFastqBytes(mQual *qual, int reverse) {
	int   length = qual->length;
	int  *data = qual->data;
	char *string = mCalloc(length+1, sizeof(char));
	coor_t i;
	if (reverse) {
		int constant = length - 1;
		for (i=0; i<length; i++) {
			string[constant - i] = data[i];
		}
	} else {
		for (i=0; i<length; i++) {
			string[i] = data[i];
		}
	}
	return string;
}

char* mFastqString(mQual *qual, int reverse) {
	int   length = qual->length;
	int  *data = qual->data;
	char *string = mCalloc(length+1, sizeof(char));
	coor_t i;
	if (reverse) {
		int constant = length - 1;
		for (i=0; i<length; i++) {
			string[constant - i] = data[i]+33;
		}
	} else {
		for (i=0; i<length; i++) {
			string[i] = data[i]+33;
		}
	}
	return string;
}

void mBreakAndWriteQual(FILE *stream, mQual *qual, coor_t chunk) {
	char *def, *dup, *brk;
	coor_t i;
	int count;

	def  = (char*) mMalloc((strlen(qual->def)+2*((int)log10(qual->length/chunk)) + 3)*sizeof(char));
	dup  = (char*) mMalloc((strlen(qual->def) + 1)*sizeof(char));
	strcpy(dup, qual->def);
	brk = strpbrk(dup, " \t");
	if (brk != NULL) {
		*brk = '\0';
	}
	for (i=0; i<qual->length; i+=chunk) {
		mQual *sub = mSubQual(qual, i, chunk);
		sprintf(def, "%s:%ld-%ld", dup, i, i+sub->length-1);
		sub->def  = def;
		mWriteQual(stream, sub);
		count++;
	}
	mFree(def);
	mFree(dup);
}

void mSplitQualToFiles(char *prefix, mQual *qual, coor_t chunk) {
	FILE *stream;
	char *def, *dup, *brk, *filename;
	coor_t i;
	int count;

	filename  = (char*) mMalloc((strlen(prefix)+((int)log10(qual->length/chunk)+1) + 2)*sizeof(char));
	def  = (char*) mMalloc((strlen(qual->def)+2*((int)log10(qual->length/chunk)) + 3)*sizeof(char));
	dup  = (char*) mMalloc((strlen(qual->def) + 1)*sizeof(char));
	strcpy(dup, qual->def);
	brk = strpbrk(dup, " \t");
	if (brk != NULL) {
		*brk = '\0';
	}
	count = 0;
	for (i=0; i<qual->length; i+=chunk) {
		mQual *sub = mSubQual(qual, i, chunk);
		sprintf(def, "%s:%ld-%ld", dup, i, i+sub->length-1);
		sprintf(filename, "%s.%d", prefix, count);
		sub->def  = def;
		if ((stream = fopen(filename, "w")) == NULL) {
			mDie("Cannot open %s for writing", filename);
		}
		mWriteQual(stream, sub);
		fclose(stream);
		count++;
	}
	mFree(def);
	mFree(dup);
	mFree(filename);
}
