#include <limits.h>
#include "mDNA.h"
#include "mQual.h"
#include <argtable2.h>

int main(int argc, char* argv[]) {
	mDNA *dna = (mDNA*) mMalloc(sizeof(mDNA));
	mQual *qual = (mQual*) mMalloc(sizeof(mQual));
	FILE *qual_stream, *fasta_stream;
	int   min_qual;
	int   soft;

	int              argcount = 0;
	int              nerrors;
	void           **argtable;

	struct arg_str  *fasta_file;
	struct arg_str  *qual_file;
	struct arg_int  *threshold;
	struct arg_lit  *softmask;
	struct arg_lit  *help;
	struct arg_end  *end;

	fasta_file = arg_str1("f", "fasta",     "<file>", "input fasta file");
	qual_file  = arg_str1("q", "qual",      "<file>", "input quality file");
	threshold  = arg_int0("t", "threshold", "<n>",    "minimum quality value to include (default: >=15)");
	softmask   = arg_lit0("s", "soft",                "softmask the query: just make it lower case");
	help       = arg_lit0("h", "help",                "print this help and exit");
	end                 = arg_end(6); /* this needs to be even, otherwise each element in end->parent[] crosses an 8-byte boundary */

	argtable          = (void**) mMalloc(6*sizeof(void*));
	argtable[argcount++] = fasta_file;
	argtable[argcount++] = qual_file;
	argtable[argcount++] = threshold;
	argtable[argcount++] = softmask;
	argtable[argcount++] = help;
	argtable[argcount++] = end;

	threshold->ival[0] = 15;

	if (arg_nullcheck(argtable) != 0) {
		mDie("insufficient memory");
	}
	nerrors = arg_parse(argc, argv, argtable);

	if (help->count > 0) {
		fprintf(stdout, "Usage: maskLowQualityBases");
		arg_print_syntax(stdout, argtable, "\n");
		arg_print_glossary(stdout, argtable, "  %-25s %s\n");
		mQuit("");
	}

	if (nerrors > 0) {
		arg_print_errors(stderr, end, "maskLowQualityBases");
		fprintf(stderr, "try using -h\n");
		mQuit("");
	}

	/* open streams */

	if ((fasta_stream = fopen(fasta_file->sval[0], "r")) == NULL) {
		mDie("Cannot open %s for reading", fasta_file->sval[0]);
	}
	if ((qual_stream = fopen(qual_file->sval[0], "r")) == NULL) {
		mDie("Cannot open %s for reading", qual_file->sval[0]);
	}

	min_qual = threshold->ival[0];
	soft     = softmask->count;

	for (;;) {
		int status;
		int i;
		mReadDNA(fasta_stream, dna);
		status = mReadQual(qual_stream, qual); 
		while (strncmp(dna->def, qual->def, MIN(strlen(dna->def), strlen(qual->def))) != 0) {
			mFreeQual(qual);
			status = mReadQual(qual_stream, qual);
		}
		for (i=0; i<qual->length; i++) {
			if (qual->data[i] < min_qual) {
				dna->data[i] = (soft)?(tolower(dna->data[i])):'N';
			}
		}
		mWriteDNA(stdout, dna);
		mFreeQual(qual);
		mFreeDNA(dna);
		if (status == END_OF_QUAL) break; /* last entry */
	}
	fclose(fasta_stream);
	fclose(qual_stream);
	exit(0);
}
