#include <limits.h>
#include <argtable2.h>
#include "bam.h"
#include "sam.h"
#include "mQual.h"
#include "mBamVector.h"
#include "zoeTools.h"

#define ADD_QUALITY       (1)
#define INSERT_SIZE       (2)
#define FILTER            (3)
#define MERGE             (4)
#define SPLIT_INSERT_SIZE (5)
#define COVERAGE          (6)

#define min(a,b)         ((a<b)?a:b)

void mCatFile(samfile_t *input, samfile_t *output);
void mFilterFile(samfile_t *input, samfile_t *output, int32_t min_length, int32_t max_distance, int32_t max_clip, int uniqbesthit_only, int besthit_only);
void mAddQuality(samfile_t *input, samfile_t *output, char *qualfile);
void mUpdateInsertSize(samfile_t *input, samfile_t *output, int insert_size, int insert_stdev, const char *insertinfo);
void mEstimateCoverageOnFile(samfile_t *input);
void mInitCoverage();

void mWriteBestHitBamPool(samfile_t *stream, mBamPool *pool);
void mWriteUniqueBestHitBamPool(samfile_t *stream, mBamPool *pool);
int  mProcessBamVectors(mBamVector **bamvector, samfile_t *output, int insert_size, int insert_stdev);
void mSplitOnInsertSize(samfile_t *input, samfile_t **outfiles, zoeHash lib2insert);
zoeHash     mReadIntegerHash(const char *filename);
samfile_t **mOpenMultipleOutputs(zoeHash lib2insert, const char *outprefix, const char *outmode, int *maxinsertsize);
void        mEmptyZoeHash(zoeHash hash);

bam_header_t  *header;
float        **coverage;
int           *covered;

int main(int argc, char* argv[]) {

	int i;

	/* common variables */

	int              mode;
	char            *qualfile;
	char            *infile;
	char            *headerfile;

	samfile_t       *input  = NULL;
	samfile_t       *output = NULL;

	char             inmode[6];
	char             outmode[6];

	/* mode specific variables */

	int              insert_size  = 0;
	int32_t          max_clip     = 0;
	int32_t          max_distance = 0;
	int32_t          min_length   = 0;

	/* argtable related */

	void           **argtable;
	struct arg_file *arg_samfile;
	struct arg_str  *arg_mode;
	struct arg_int  *arg_insertsize;
	struct arg_str  *arg_insertfile;
	struct arg_int  *arg_minpercentid;
	struct arg_int  *arg_minlength;
	struct arg_int  *arg_minqfrac;
	struct arg_lit  *arg_besthitonly;
	struct arg_lit  *arg_uniqbesthitonly;
	struct arg_str  *arg_qualfile;
	struct arg_str  *arg_headfile;
	struct arg_str  *arg_outprefix;
	struct arg_lit  *arg_bamout;
	struct arg_lit  *arg_samin;
	struct arg_lit  *arg_uncompressed;
	struct arg_lit  *arg_write_header;
	struct arg_lit  *help;
	struct arg_end  *end;
	int              nerrors;
	int              argcount=0;

	arg_mode            = arg_str1("m",  NULL,     NULL,           "mode to process SAM/BAM files (merge, filter, addquality, insertsize, splitinsertsize)\n\n"
									"\tmerge            :  merge multiple SAM/BAM files and write to stdout\n"
									"\tcoverage         :  estimate coverage of each reference sequence (only one input bam/sam file)\n"
									"\tfilter           :  filter out alignments that do not satisfy requirements\n"
									"\taddquality       :  add quality string for the reported region of read\n"
									"\tinsertsize       :  estimate insert size of mate-pairs and update the field\n"
									"\tsplitinsertsize  :  split bam/sam files based on estimated insert size\n\n"
									"Options for I/O of BAM/SAM files (same meaning as in 'samtools view'):\n");
	arg_samin           = arg_lit0("S",  NULL,                     "input is SAM (default: false)");
	arg_bamout          = arg_lit0("b",  NULL,                     "output BAM (default: false)");
	arg_uncompressed    = arg_lit0("u",  NULL,                     "uncompressed BAM output (force -b) (default: false)");
	arg_write_header    = arg_lit0("h",  NULL,                     "print header for the SAM output (default: false)");
	arg_headfile        = arg_str0("t",  NULL,     "<file>",       "list of reference names and lengths (required if input contains multiple BAM/SAM files)\n\nfilter: filter out alignments that do not satisfy requirements\n");

	arg_minlength       = arg_int0("l",  NULL,     NULL,           "min. length of alignment (required)");
	arg_minpercentid    = arg_int0("p",  NULL,     NULL,           "min. percent identity of alignment, between 0 and 100; requires NM field to be present (required)");
	arg_minqfrac        = arg_int0("z",  NULL,     NULL,           "min. percent of the query that must be aligned, between 0 and 100 (required)\n\n"
									"  The following special filters require:\n"
									"    (1) the alignments to be sorted by name,\n"
									"    (2) AS field to be present.\n"
									"  You can usually achieve this by \"samtools sort -n -o - input.bam dummy | msamtools -m filter --besthit -\"\n");
	arg_besthitonly     = arg_lit0(NULL, "besthit",                "keep only the highest scoring hit per read (default: false)");
	arg_uniqbesthitonly = arg_lit0(NULL, "uniqhit",                "keep only the highest scoring hit per read, if it is unique (default: false)\n\n"
									"addquality: add quality string for the reported region of read\n");

	arg_qualfile        = arg_str0("q",  NULL,     "<file>",       "input quality file - order of reads in this file should be the same as in SAM/BAM files (required)\n\n"
									"insertsize: estimate insert size of mate-pairs and update the field\n");

	arg_insertsize      = arg_int0("i",  NULL,     NULL,           "expected insert size (one of -i or --insert is required)");
	arg_insertfile      = arg_str0(NULL, "insert", "<file>",       "tab delimited file of machine:lane prefix with insert size (one of -i or --insert is required)\n\n"
									"splitinsertsize: split bam/sam files based on estimated insert size (requires --insert)\n");


	arg_outprefix       = arg_str0("o",  NULL,     NULL,           "prefix for the output files (required)\n\n"
									"General options:\n");

	help                = arg_lit0(NULL, "help",                   "print this help and exit");
	arg_samfile         = arg_filen(NULL, NULL,    "<bamfile>", 1, argc+3,   "input SAM/BAM file");

	end                 = arg_end(8); /* this needs to be even, otherwise each element in end->parent[] crosses an 8-byte boundary */

	argtable          = (void**) mCalloc(18, sizeof(void*));
	argtable[argcount++] = arg_mode;
	argtable[argcount++] = arg_samin;
	argtable[argcount++] = arg_bamout;
	argtable[argcount++] = arg_uncompressed;
	argtable[argcount++] = arg_write_header;
	argtable[argcount++] = arg_headfile;
	argtable[argcount++] = arg_minlength;
	argtable[argcount++] = arg_minpercentid;
	argtable[argcount++] = arg_minqfrac;
	argtable[argcount++] = arg_besthitonly;
	argtable[argcount++] = arg_uniqbesthitonly;
	argtable[argcount++] = arg_qualfile;
	argtable[argcount++] = arg_insertsize;
	argtable[argcount++] = arg_insertfile;
	argtable[argcount++] = arg_outprefix;
	argtable[argcount++] = help;
	argtable[argcount++] = arg_samfile;
	argtable[argcount++] = end;

	/* defaults */

	/* parse command line */

	if (arg_nullcheck(argtable) != 0) {
		mDie("insufficient memory");
	}
	nerrors = arg_parse(argc, argv, argtable);

	if (help->count > 0 || arg_samfile->count == 0) {
		fprintf(stdout, "\nmsamtools: Tools you wish samtools had included!\n");
		fprintf(stdout,   "           Part of the awesome maui library in Smash\n");
		fprintf(stdout,   "           Smash: Making your life easy, step by step (TM)\n\n");
		fprintf(stdout, "Usage: msamtools");
		arg_print_syntax(stdout, argtable, "\n");
		fprintf(stdout, "\nChoosing the right mode:\n\n");
		arg_print_glossary(stdout, argtable, "  %-25s %s\n");
		mQuit("");
	}

	if (nerrors > 0) {
		arg_print_errors(stderr, end, "msamtools");
		fprintf(stderr, "try using --help\n");
		mQuit("");
	}

	if (arg_samfile->count > 1 && arg_headfile->count == 0) {
		fprintf(stderr, "-t required when using multiple input files!\n");
		fprintf(stdout, "Usage: msamtools");
		arg_print_syntax(stdout, argtable, "\n");
		arg_print_glossary(stdout, argtable, "  %-25s %s\n");
		mQuit("");
	}

	/* figure out which mode we run under */

	if (strcmp(arg_mode->sval[0], "addquality") == 0) {
		mode = ADD_QUALITY;
		if (arg_qualfile->count == 0) {
			fprintf(stderr, "-m addquality requires quality file through -q\n");
			fprintf(stdout, "Usage: msamtools");
			arg_print_syntax(stdout, argtable, "\n");
			arg_print_glossary(stdout, argtable, "  %-25s %s\n");
			mQuit("");
		}
	} else if (strcmp(arg_mode->sval[0], "insertsize") == 0) {
		mode = INSERT_SIZE;
		if (arg_insertsize->count == 0 && arg_insertfile->count == 0) {
			fprintf(stderr, "-m insertsize requires expected insert size through -i or --insert\n");
			fprintf(stdout, "Usage: msamtools");
			arg_print_syntax(stdout, argtable, "\n");
			arg_print_glossary(stdout, argtable, "  %-25s %s\n");
			mQuit("");
		}
		if (arg_insertsize->count > 0) {
			insert_size = arg_insertsize->ival[0];
		} else {
			insert_size = 0;
		}
	} else if (strcmp(arg_mode->sval[0], "splitinsertsize") == 0) {
		mode = SPLIT_INSERT_SIZE;
		if (arg_outprefix->count == 0 && arg_insertfile->count == 0) {
			fprintf(stderr, "-m splitinsertsize requires --outprefix and expected insert size through --insert\n");
			fprintf(stdout, "Usage: msamtools");
			arg_print_syntax(stdout, argtable, "\n");
			arg_print_glossary(stdout, argtable, "  %-25s %s\n");
			mQuit("");
		}
	} else if (strcmp(arg_mode->sval[0], "filter") == 0) {
		mode = FILTER;
		if (arg_minlength->count == 0 || arg_minpercentid->count == 0 || arg_minqfrac->count == 0) {
			fprintf(stdout, "--mode filter needs -l, -p and -z\n");
			fprintf(stdout, "Usage: msamtools");
			arg_print_syntax(stdout, argtable, "\n");
			arg_print_glossary(stdout, argtable, "  %-25s %s\n");
			mQuit("");
		} else {
			int32_t pid   = arg_minpercentid->ival[0];
			int32_t qfrac = arg_minqfrac->ival[0];

			min_length   = (int32_t) arg_minlength->ival[0];
			if (pid < 0 || pid > 100 || qfrac < 0 || qfrac > 100) {
				fprintf(stdout, "-p and -z must be in the range [0,100]\n");
				fprintf(stdout, "Usage: msamtools");
				arg_print_syntax(stdout, argtable, "\n");
				arg_print_glossary(stdout, argtable, "  %-25s %s\n");
				mQuit("");
			}
			max_distance = 100 - pid;
			max_clip     = 100 - qfrac;
		}
	} else if (strcmp(arg_mode->sval[0], "merge") == 0) {
		mode = MERGE;
	} else if (strcmp(arg_mode->sval[0], "coverage") == 0) {
		mode = COVERAGE;
		if (arg_samfile->count > 1) {
			fprintf(stdout, "--mode coverage requires ONE bam/sam input file\n");
			fprintf(stdout, "Usage: msamtools");
			arg_print_syntax(stdout, argtable, "\n");
			arg_print_glossary(stdout, argtable, "  %-25s %s\n");
			mQuit("");
		}
	} else if (strcmp(arg_mode->sval[0], "test") == 0) {
		mBamPool *pool = (mBamPool*) mCalloc(1, sizeof(mBamPool));
		mInitBamPool(pool, 10);
		pool_current(pool);
		mFreeBamPool(pool);
		mFree(pool);
		mode = -1;
	} else {
		mode = -1;
		fprintf(stderr, "Unknown mode: %s", arg_mode->sval[0]);
		fprintf(stdout, "Usage: msamtools");
		arg_print_syntax(stdout, argtable, "\n");
		arg_print_glossary(stdout, argtable, "  %-25s %s\n");
		mQuit("");
	}

	/* alloc memory for all file names */

	infile = (char*) mCalloc(256, sizeof(char));

	qualfile = (char*) mCalloc(256, sizeof(char));
	if (arg_qualfile->count > 0)
		strcpy(qualfile, arg_qualfile->sval[0]);

	header     = NULL;
	headerfile = NULL;

	if (arg_headfile->count > 0) {
		headerfile = (char*) mCalloc(256, sizeof(char));
		strcpy(headerfile, arg_headfile->sval[0]);
		header = sam_header_read2(headerfile);
	}

	/* set input mode */

	strcpy(inmode, "r");
	if (arg_samin->count == 0) 
		strcat(inmode, "b");

	/* set output mode */

	strcpy(outmode, "w");
	if (arg_uncompressed->count > 0) 
		strcat(outmode, "bu");
	else if (arg_bamout->count > 0) 
		strcat(outmode, "b");
	else if (arg_write_header->count > 0) 
		strcat(outmode, "h");

{ /* localize variables */
	zoeHash     lib2insert = NULL;
	samfile_t **outfiles = NULL;
	int         maxinsertsize;
	for (i=0; i<arg_samfile->count; i++) {

		bam_header_t *header_bak = NULL;
		strcpy(infile, arg_samfile->filename[i]);

		/* open input file */
		/* since you pass headerfile, if there is a headerfile, it will be used */

		input = samopen(infile, inmode, headerfile);
		if (input == NULL) 
			mDie("Cannot open %s for reading", infile);

		/* if there is a headerfile through -t, take it and replace the read header with it! */
		/* else make sure header was read through samopen() */

		if (headerfile != NULL) {
			header_bak = input->header;
			input->header = header;
		} else {
			if (input->header == 0)
				mDie("No header found in %s, and no header given through -t", infile);
			else
				header = input->header;
		}

		/* header processing when you read the first file */

		if (i==0) {
			if (mode == SPLIT_INSERT_SIZE) {
				lib2insert = mReadIntegerHash(arg_insertfile->sval[0]);
				outfiles = mOpenMultipleOutputs(lib2insert, arg_outprefix->sval[0], outmode, &maxinsertsize);
			} else if (mode == COVERAGE) {
				mInitCoverage();
			} else {
				output = samopen("-", outmode, header);
			}
		}

		switch (mode) {
			case ADD_QUALITY:
				mAddQuality(input, output, qualfile);
				break;
			case INSERT_SIZE:
				mUpdateInsertSize(input, output, insert_size, insert_size/10, arg_insertfile->sval[0]);
				break;
			case SPLIT_INSERT_SIZE:
				mSplitOnInsertSize(input, outfiles, lib2insert);
				break;
			case FILTER:
				mFilterFile(input, output, min_length, max_distance, max_clip, arg_uniqbesthitonly->count > 0, arg_besthitonly->count > 0);
				break;
			case MERGE:
				mCatFile(input, output);
				break;
			case COVERAGE:
				mEstimateCoverageOnFile(input);
				break;
			default:
				break;
		}

		if (mode == COVERAGE) {
			mCoverageToQual();
		}

		fprintf(stderr, "%s\n", arg_samfile->filename[i]);
		if (headerfile != NULL) {
			input->header = header_bak;
		}
		samclose(input);
	}

	if (mode == SPLIT_INSERT_SIZE) {
		for (i=0; i <= maxinsertsize; i++) {
			if (outfiles[i] != NULL)
				samclose(outfiles[i]);
		}
		mFree(outfiles);
		mEmptyZoeHash(lib2insert);
		zoeDeleteHash(lib2insert);
	} else {
		samclose(output);
	}

} /* end localize variables */

	if (headerfile != NULL) {
		mFree(headerfile);
		bam_header_destroy(header);
	}
	mFree(infile);
	mFree(qualfile);
	arg_freetable(argtable, argcount);
	mFree(argtable);
	return 0;
}

void mCatFile(samfile_t *input, samfile_t *output) {
	bam1_t *b = bam_init1();
	while (samread(input, b) >= 0) {
		samwrite(output, b);
	}
	bam_destroy1(b);
}

/****************************
 * NOTE on 27.11.2010:
 * I had a separate mode for besthit selection since that will involve
 * keeping a list of scores, processing it and then printing.
 * But I did a test on BFAST alignment of MH6, and the difference of 
 * using a list and processing it was just 3m47s from 3m44s. So I decided
 * to make it the standard way to process groups. Both -m filter and -m filter -b
 * will go through the same procedure, so no need to maintain two versions of code
 */

void mFilterFile(samfile_t *input, samfile_t *output, int32_t min_length, int32_t max_distance, int32_t max_clip, int uniqbesthit_only, int besthit_only) {

	int       pool_limit = 64;
	bam1_t   *current;
	mBamPool *pool = (mBamPool*) mCalloc(1, sizeof(mBamPool));
	char     *prev_read = NULL;

	/* features to filter on */

	uint8_t  *nm;
	int32_t   edit;
	int32_t   alength, qlength, qclip;

	/* write function */

	void      (*writer)(samfile_t*, mBamPool*);

	if (uniqbesthit_only)
		writer = mWriteUniqueBestHitBamPool;
	else if (besthit_only)
		writer = mWriteBestHitBamPool;
	else
		writer = mWriteBamPool;

	/* init pool */

	mInitBamPool(pool, pool_limit);
	current = pool_current(pool);

	while (samread(input, current) >= 0) {
		if (prev_read != NULL && strcmp(bam1_qname(current), prev_read) != 0) {
			writer(output, pool);
			mReOriginateBamPool(pool);
		}

		/* get the alignment length, query length, query clipped length */

		bam_cigar2details(&current->core, bam1_cigar(current), &alength, &qlength, &qclip);
#ifdef DEBUG
			fprintf(stdout, "qlen=%d; qclip=%d; alen=%d\n", qlength, qclip, alength );
#endif

		/* check for minimum alignment length */

		if (alength < min_length) continue;

		/* check for the minimum fraction of query aligned */

		if (100*qclip > max_clip*qlength) continue;

		/* check the edit distance */

		nm = bam_aux_get(current, "NM");
		if (nm) {
			edit = bam_aux2i(nm);
#ifdef DEBUG
			fprintf(stdout, "edit=%d; max=%d; lhs=%d; rhs=%d\n", edit, max_distance, 100*edit, alength*max_distance, 100*edit );
#endif
			if (100*edit > alength*max_distance) continue;
		}
		prev_read = bam1_qname(current);
		current = mAdvanceBamPool(pool);
	}
	writer(output, pool);
	mFreeBamPool(pool);
	mFree(pool);
}

/* BEGIN COVERAGE RELATED STUFF */

void mUpdateCoverageForAlignment(bam1_t *bam, float cov) {
	int i;
	int tid   = bam->core.tid;
	int start = bam->core.pos;
	int end   = bam_calend(&bam->core, bam1_cigar(bam));

#ifdef DEBUG
	fprintf(stderr, "Tagging target %d (%s) with %f: from %d to %d\n", tid, header->target_name[tid], cov, start, end);
#endif

	/* check before you write: read takes less time than write, and write happens only n_targets times MAX */

	if (!covered[tid]) 
		covered[tid] = 1; 
	for (i=start; i<=end; i++) {
		coverage[tid][i] += cov;
	}
}

void mEstimateCoverageOnPool(mBamPool *pool) {
	int i;
	int n;
	float cov;
	bam1_t **elem = pool->elem;
	int origin    = pool->origin;
	int current   = pool->current;
	int limit     = pool->limit;
	if (origin <= current) {
		n = current - origin;
		cov = 1.0/n;
		for (i=origin; i<current; i++) 
			mUpdateCoverageForAlignment(elem[i], cov);
	} else {
		n = current + limit - origin;
		cov = 1.0/n;
		for (i=origin; i<limit; i++) 
			mUpdateCoverageForAlignment(elem[i], cov);
		for (i=0; i<current; i++) 
			mUpdateCoverageForAlignment(elem[i], cov);
	}
}

void mInitCoverage() {
	int     tid;
	int     n_targets = header->n_targets;

	coverage  = (float**) mCalloc(n_targets, sizeof(float*));
	covered   = (int*) mCalloc(n_targets, sizeof(int));

	for (tid=0; tid<n_targets; tid++) {
		int tlen  = header->target_len[tid];
		coverage[tid] = (float*) mCalloc(tlen, sizeof(float));
	}
}

void mCoverageToQual() {
	int     i;
	int     tid;
	int     n_targets = header->n_targets;
	int     tlen;
	int    *data;
	float  *cov;
	mQual  *qual = (mQual*) mCalloc(1, sizeof(mQual));
	for (tid=0; tid<n_targets; tid++) {
		if (covered[tid] != 1) continue;
		tlen  = header->target_len[tid];
		mInitQual(qual, tlen);
		data = qual->data;
		cov  = coverage[tid];
		for (i=0; i<tlen; i++) {
			data[i] = (int)cov[i];
		}
		qual->def = header->target_name[tid];
		mWriteQual(stdout, qual);
		qual->def = NULL;
		mFreeQual(qual);
	}
	mFree(qual);
}

void mEstimateCoverageOnFile(samfile_t *input) {

	int       pool_limit = 64;
	bam1_t   *current;
	mBamPool *pool = (mBamPool*) mCalloc(1, sizeof(mBamPool));
	char     *prev_read = NULL;

	/* init pool */

	mInitBamPool(pool, pool_limit);
	current = pool_current(pool);

	while (samread(input, current) >= 0) {
		if (prev_read != NULL && strcmp(bam1_qname(current), prev_read) != 0) {
			mEstimateCoverageOnPool(pool);
			mReOriginateBamPool(pool);
		}
		prev_read = bam1_qname(current);
		current   = mAdvanceBamPool(pool);
	}
	mEstimateCoverageOnPool(pool);
	mFreeBamPool(pool);
	mFree(pool);

}

/* END COVERAGE RELATED STUFF */

zoeHash mReadIntegerHash(const char *filename) {
	char     line[2048];
	char     library[128];
	int     *insert_size;
	FILE    *stream;
	zoeHash  hash;

	stream = fopen(filename, "r");
	if (stream == NULL)
		mDie("Cannot open file: %s\n", filename);

	insert_size = (int*) mCalloc(1, sizeof(int));;
	hash = zoeNewHash();
	while (fgets(line, sizeof(line), stream) != NULL) {
		if (sscanf(line, "%s\t%d", library, insert_size) == 2) {
			zoeSetHash(hash, library, insert_size);
		}
		insert_size = (int*) mCalloc(1, sizeof(int));
	}
	fclose(stream);
	mFree(insert_size);
	return hash;
}

void mEmptyZoeHash(zoeHash hash) {
	int    i;
	zoeVec vals;
	if (hash == NULL) 
		return;
	vals = zoeValsOfHash(hash); 
	for (i=0; i<vals->size; i++) {
		mFree(vals->elem[i]);
	}
	zoeDeleteVec(vals);
}

samfile_t **mOpenMultipleOutputs(zoeHash lib2insert, const char *outprefix, const char *outmode, int *maxinsertsize) {
	int         i;
	samfile_t **outfiles;
	zoeVec      inserts  = zoeValsOfHash(lib2insert);

	*maxinsertsize = INT_MIN;

	/* open output files */

	for (i=0; i < inserts->size; i++) {
		int s = *((int*) inserts->elem[i]);
		if (s > *maxinsertsize) *maxinsertsize = s;
	}
	outfiles = (samfile_t**) mCalloc(*maxinsertsize+1, sizeof(samfile_t*));
	for (i=0; i<=*maxinsertsize; i++) 
		outfiles[i] = NULL;
	for (i=0; i < inserts->size; i++) {
		int s = *((int*) inserts->elem[i]);
		if (outfiles[s] == NULL) {
			char outname[128];
			sprintf(outname, "%s.%d.bam", outprefix, s);
			outfiles[s] = samopen(outname, outmode, header);
		}
	}
	return outfiles;
}

void mSplitOnInsertSize(samfile_t *input, samfile_t **outfiles, zoeHash lib2insert) {

	/* BAM/SAM related */

	bam1_t     *current;
	int        *isize;
	char       *prev_read = NULL;
	int         pool_limit = 64;
	mBamPool   *pool = (mBamPool*) mCalloc(1, sizeof(mBamPool));
	char       *prefix = mCalloc(128, sizeof(char));
	
	/* init pool */

	mInitBamPool(pool, pool_limit);
	current = pool_current(pool);

	/* parse sam file */

	while (samread(input, current) >= 0) {
		if (prev_read != NULL && strcmp(bam1_qname(current), prev_read) != 0) {
			strcpy(prefix, prev_read);
			*index(index(prefix, ':')+1, ':') = '\0';
			isize = (int*) zoeGetHash(lib2insert, prefix);
			if (isize == NULL)
				mDie("Cannot find insert size for library %s!", prefix);
			mWriteBamPool(outfiles[*isize], pool);
			mReOriginateBamPool(pool);
		}
		prev_read = bam1_qname(current);
		current = mAdvanceBamPool(pool);
	}
	strcpy(prefix, prev_read);
	*index(index(prefix, ':')+1, ':') = '\0';
	isize = (int*) zoeGetHash(lib2insert, prefix);
	if (isize == NULL)
		mDie("Cannot find insert size for library %s!", prefix);
	mWriteBamPool(outfiles[*isize], pool);

	mFreeBamPool(pool);
	mFree(pool);
	mFree(prefix);
}

void mUpdateInsertSize(samfile_t *input, samfile_t *output, int insert_size, int insert_stdev, const char *insertinfo) {

	/* BAM/SAM related */

	bam1_t        *b;
	bam1_t        *prev = NULL;
	mBamVector   **bamvector;
	char          *prefix = mCalloc(128, sizeof(char));

	/* place holders for bam objects */

	bamvector    = (mBamVector**) mCalloc(2, sizeof(mBamVector*));
	bamvector[0] = (mBamVector*) mCalloc(1, sizeof(mBamVector));
	mInitBamVector(bamvector[0], 1000);
	bamvector[1] = (mBamVector*) mCalloc(1, sizeof(mBamVector));
	mInitBamVector(bamvector[1], 1000);

	/* parse sam file */

	b = bam_init1();

	if (insert_size == 0) {
		int    *isize;
		zoeHash lib2insert = mReadIntegerHash(insertinfo);
		while (samread(input, b) >= 0) {
			int mate = (int) bam1_qfrag(b);
			if (prev != NULL && bam1_templatecmp(b, prev) != 0) {
				strcpy(prefix, bam1_qname(prev));
				*index(index(prefix, ':')+1, ':') = '\0';
				isize = (int*) zoeGetHash(lib2insert, prefix);
				if (isize == NULL)
					mDie("Cannot find insert size for library %s!", prefix);
#ifdef DEBUG
				fprintf(stdout, "read:%s; library:%s; insertsize:%d;\n", bam1_qname(prev), prefix, *isize);
#endif
				mProcessBamVectors(bamvector, output, *isize, *isize/10);
			}
			mPushBamVector(bamvector[mate], b);
			prev = b;
			b = bam_init1();
		}
		strcpy(prefix, bam1_qname(prev));
		*index(index(prefix, ':')+1, ':') = '\0';
		isize = (int*) zoeGetHash(lib2insert, prefix);
		mProcessBamVectors(bamvector, output, *isize, *isize/10);
		mEmptyZoeHash(lib2insert);
		zoeDeleteHash(lib2insert);
	} else {
		while (samread(input, b) >= 0) {
			int mate = (int) bam1_qfrag(b);
			if (prev != NULL && bam1_templatecmp(b, prev) != 0) {
				mProcessBamVectors(bamvector, output, insert_size, insert_stdev);
			}
			mPushBamVector(bamvector[mate], b);
			prev = b;
			b = bam_init1();
		}
		mProcessBamVectors(bamvector, output, insert_size, insert_stdev);
	}

	bam_destroy1(b);
	mFreeBamVector(bamvector[0]);
	mFree(bamvector[0]);
	mFreeBamVector(bamvector[1]);
	mFree(bamvector[1]);
	mFree(bamvector);
	mFree(prefix);
}

int mCompareBam(const void *a, const void *b) {
	bam1_core_t *ca = &((bam1_t*)a)->core;
	bam1_core_t *cb = &((bam1_t*)b)->core;
	int32_t      ta = ca->tid;
	int32_t      tb = cb->tid;
	char        *na = header->target_name[ta];
	char        *nb = header->target_name[tb];
	int          la = target_taxlen(na);
	int          lb = target_taxlen(nb);
	int         len = (la<lb)?la:lb;
	return strncmp(na, nb, len) || (ta - tb) || (ca->pos - cb->pos);
}

int mCompareTID(const void *a, const void *b) {
	int ret = ((*(bam1_t**)a)->core).tid - ((*(bam1_t**)b)->core).tid;
#ifdef DEBUG
	bam1_t *ab = *(bam1_t**)a;
	bam1_t *bb = *(bam1_t**)b;
	int32_t at = (ab->core).tid;
	int32_t bt = (bb->core).tid;
	fprintf(stdout, "cmp: %d:%s vs %d:%s returns %d\n", at, header->target_name[at], bt, header->target_name[bt], ret);
#endif
	return ret;
}

bam1_t* mFindBestMate(bam1_t *b, mBamVector *bamvector) {
	int i;
	int size = bamvector->size;
	bam1_t **elem = bamvector->elem;
	int needle_taxid_len;
	int needle_tid  = b->core.tid;
	char *needle_refname = header->target_name[needle_tid];
	/*fprintf(stdout, "Looking for %d:%s\n", needle_tid, needle_refname);*/
	for (i=0; i<size; i++) {
		if (elem[i]->core.tid == needle_tid) {
			/*fprintf(stdout, "Found %d\n", needle_tid);*/
			return elem[i];
		}
	}
	needle_taxid_len = target_taxlen(needle_refname);
	for (i=0; i<size; i++) {
		char *haystack_refname = header->target_name[elem[i]->core.tid];
		int haystack_taxid_len = target_taxlen(haystack_refname);
		if (strncmp(haystack_refname, needle_refname, min(haystack_taxid_len, needle_taxid_len)) == 0) {
			/*fprintf(stdout, "Found %s\n", ref);*/
			return elem[i];
		}
	}
	return NULL;
}

void mGetOptimalPairing(mBamVector *vec0, mBamVector *vec1, int insert_size, int insert_stdev) {
	int i, j;

	int size0 = vec0->size;
	int size1 = vec1->size;

	int32_t  *pos0  = (int32_t*) mCalloc(size0, sizeof(int32_t));
	int      *rev0  = (int*)     mCalloc(size0, sizeof(int));
	int32_t  *pos1  = (int32_t*) mCalloc(size1, sizeof(int32_t));
	int      *rev1  = (int*)     mCalloc(size1, sizeof(int));

	/* init arrays for the zero-th */

#ifdef DEBUG
	fprintf(stdout, "Trying pairing for %d vs %d\n", vec0->size, vec1->size);
#endif

	for (i=0; i<size0; i++) {
		bam1_core_t core;
		core     = vec0->elem[i]->core;
		rev0[i]  = ((core.flag&BAM_FREVERSE)==BAM_FREVERSE);
		pos0[i]  = (rev0[i])?(int32_t)bam_calend(&core, bam1_cigar(vec0->elem[i])):core.pos;
	}

	/* search zero-th for first */

	for (i=0; i<size1; i++) {
		int best = -1;
		int best_same = -1, best_opp = -1;
		int best_same_dist = INT_MAX, best_opp_dist = INT_MAX;
		int best_same_abs_dev = INT_MAX, best_opp_abs_dev = INT_MAX;
		int rev;
		int32_t pos;
		bam1_core_t *core;
		core     = &vec1->elem[i]->core;
		rev      = ((core->flag&BAM_FREVERSE)==BAM_FREVERSE);
		pos      = (rev1[i])?(int32_t)bam_calend(core, bam1_cigar(vec1->elem[i])):core->pos;
		rev1[i]  = rev;
		pos1[i]  = pos;
		for (j=0; j<size0; j++) {
			int dist = pos0[j]-pos;
			int abs_dev = abs(insert_size-abs(dist));
			if (rev == rev0[j]) {
				if (abs_dev < best_same_abs_dev) {
					best_same = j;
					best_same_dist = dist;
					best_same_abs_dev = abs_dev;
				}
			} else {
				if (abs_dev < best_opp_abs_dev) {
					best_opp = j;
					best_opp_dist = dist;
					best_opp_abs_dev = abs_dev;
				}
			}
		}

		/*******
		 * If an opposite direction pairing is more optimal than same direction pairing, then choose it.
		 * This could be an outie, but still this is the optimal, so it doesn't matter.
		 * If an opposite direction pairing is not an outie, you can cut some slack for 
		 * the proper pairing even if it is not as good as a same direction pairing.
		 */

		if (best_opp_abs_dev < best_same_abs_dev) {
			best = best_opp;
			core->isize = best_opp_dist;
			if (rev != (best_opp_dist>0)) core->flag |= BAM_FPROPER_PAIR;
		} else if (best_opp_abs_dev <= 100000 && rev != (best_opp_dist>0) &&                          /* neither same direction (condition 1) nor outie (condition 2) */
			  (best_opp_abs_dev < 3.5*insert_stdev || best_opp_abs_dev < 2*best_same_abs_dev)) {  /* this is the slack - two parameters to tune */
			fprintf(stderr, "%-40s at %20s:%08d preferred %08d(isize:%8d) over %08d(isize:%8d) due to proper pairing!\n", bam1_qname(vec1->elem[i]), header->target_name[vec1->elem[i]->core.tid], core->pos, vec0->elem[best_opp]->core.pos, best_opp_dist, vec0->elem[best_same]->core.pos, best_same_dist);
			best = best_opp;
			core->isize = best_opp_dist;
			core->flag |= BAM_FPROPER_PAIR;
		} else {
			best = best_same;
			core->isize = best_same_dist;
		}
#ifdef DEBUG
		fprintf(stdout, "Pairing %d and %d\n", i, best);
#endif
		core->mtid = core->tid;
		core->mpos = vec0->elem[best]->core.pos;
		if (core->pos < core->mpos) core->flag |= BAM_FREAD1;
		else                        core->flag |= BAM_FREAD2;
		if (rev0[best])             core->flag |= BAM_FMREVERSE;
#ifdef DEBUG
		fprintf(stdout, "%d vs %d\n", core->pos, core->mpos);
#endif
	}

	/* search first for zero-th */

	for (i=0; i<size0; i++) {
		int best = -1;
		int best_same = -1, best_opp = -1;
		int best_same_dist = INT_MAX, best_opp_dist = INT_MAX;
		int best_same_abs_dev = INT_MAX, best_opp_abs_dev = INT_MAX;
		int     rev = rev0[i];
		int32_t pos = pos0[i];
		bam1_core_t *core;
		core     = &vec0->elem[i]->core;
		for (j=0; j<size1; j++) {
			int dist = pos1[j]-pos;
			int abs_dev = abs(insert_size-abs(dist));
			if (rev == rev1[j]) {
				if (abs_dev < best_same_abs_dev) {
					best_same = j;
					best_same_dist = dist;
					best_same_abs_dev = abs_dev;
				}
			} else {
				if (abs_dev < best_opp_abs_dev) {
					best_opp = j;
					best_opp_dist = dist;
					best_opp_abs_dev = abs_dev;
				}
			}
		}

		/*******
		 * If an opposite direction pairing is more optimal than same direction pairing, then choose it.
		 * This could be an outie, but still this is the optimal, so it doesn't matter.
		 * If an opposite direction pairing is not an outie, you can cut some slack for 
		 * the proper pairing even if it is not as good as a same direction pairing.
		 */

		if (best_opp_abs_dev < best_same_abs_dev) {
			best = best_opp;
			core->isize = best_opp_dist;
			if (rev != (best_opp_dist>0)) core->flag |= BAM_FPROPER_PAIR;
		} else if (best_opp_abs_dev <= 100000 && rev != (best_opp_dist>0) &&                          /* neither same direction (condition 1) nor outie (condition 2) */
			  (best_opp_abs_dev < 3.5*insert_stdev || best_opp_abs_dev < 2*best_same_abs_dev)) {  /* this is the slack - two parameters to tune */
			fprintf(stderr, "%-40s at %20s:%08d preferred %08d(isize:%8d) over %08d(isize:%8d) due to proper pairing!\n", bam1_qname(vec0->elem[i]), header->target_name[vec0->elem[i]->core.tid], core->pos, vec1->elem[best_opp]->core.pos, best_opp_dist, vec1->elem[best_same]->core.pos, best_same_dist);
			best = best_opp;
			core->isize = best_opp_dist;
			core->flag |= BAM_FPROPER_PAIR;
		} else {
			best = best_same;
			core->isize = best_same_dist;
		}
#ifdef DEBUG
		fprintf(stdout, "Pairing %d and %d\n", i, best);
#endif
		core->mtid = core->tid;
		core->mpos = vec1->elem[best]->core.pos;
		if (core->pos < core->mpos) core->flag |= BAM_FREAD1;
		else                        core->flag |= BAM_FREAD2;
		if (rev1[best])             core->flag |= BAM_FMREVERSE;
#ifdef DEBUG
		fprintf(stdout, "%d vs %d\n", core->pos, core->mpos);
#endif
	}
	mFree(pos0);
	mFree(rev0);
	mFree(pos1);
	mFree(rev1);
}

int mProcessBamVectors(mBamVector **bamvector, samfile_t *output, int insert_size, int insert_stdev) {
	int i;
	int index0, index1;
	int32_t  curr_tid;
	mBamVector *fwd, *rev;

	bam1_t  **elem0;
	int       size0 = bamvector[0]->size;
	int32_t  *tid0  = (int32_t*) mCalloc(size0, sizeof(int32_t));

	bam1_t  **elem1;
	int       size1 = bamvector[1]->size;
	int32_t  *tid1  = (int32_t*) mCalloc(size1, sizeof(int32_t));

	/* sort the two vectors */

#ifdef DEBUG
	fprintf(stdout, "BEFOR SORT:\n");
	mWriteBamVector(output, bamvector[0]);
	mWriteBamVector(output, bamvector[1]);
#endif

	qsort(bamvector[0]->elem, bamvector[0]->size, sizeof(bam1_t*), mCompareTID);
	qsort(bamvector[1]->elem, bamvector[1]->size, sizeof(bam1_t*), mCompareTID);

#ifdef DEBUG
	fprintf(stdout, "AFTER SORT:\n");
	mWriteBamVector(output, bamvector[0]);
	mWriteBamVector(output, bamvector[1]);
#endif

	/* store tid's */

	elem0 = bamvector[0]->elem;
	for (i=0; i<size0; i++) tid0[i] = elem0[i]->core.tid;
	elem1 = bamvector[1]->elem;
	for (i=0; i<size1; i++) tid1[i] = elem1[i]->core.tid;

	/* start in the beginning */

	index0 = index1 = 0;
	fwd = (mBamVector*) mCalloc(1, sizeof(mBamVector));
	mInitBamVector(fwd, 8);
	rev = (mBamVector*) mCalloc(1, sizeof(mBamVector));
	mInitBamVector(rev, 8);

	do {
		/* step forward in both until you hit the same tid */

		do {
			while (tid0[index0] < tid1[index1] && index0 < size0) index0++;
			while (tid1[index1] < tid0[index0] && index1 < size1) index1++;
			if (index0 >= size0 || index1 >= size1) break;
		} while (tid0[index0] != tid1[index1]);

		if (index0 >= size0 || index1 >= size1) break;

		/* make two lists containing the bam's of the current tid */

		curr_tid = tid0[index0];
		fwd->size = rev->size = 0;

		while (tid0[index0] == curr_tid && index0 < size0) {
			mPushBamVector(fwd, elem0[index0]);
			index0++;
		}

		while (tid1[index1] == curr_tid && index1 < size1) {
			mPushBamVector(rev, elem1[index1]);
			index1++;
		}

		/* process the two lists */

#ifdef DEBUG
		fprintf(stdout, "Processing %d:%s\n", curr_tid, header->target_name[curr_tid]);
#endif
		if (fwd->size > 0 && rev->size > 0) mGetOptimalPairing(fwd, rev, insert_size, insert_stdev);

	} while (index0 < size0 && index1 < size1);

	mFreeBamVector(fwd);
	mFreeBamVector(rev);
	mFree(tid0);
	mFree(tid1);

	mWriteBamVector(output, bamvector[0]);
	mWriteBamVector(output, bamvector[1]);

#ifdef MAUI_OLD
	/* Deal with mate pairing flags based on mating information */

	for (i=0; i<bamvector[0]->size; i++) {
		bam1_t *b    = bamvector[0]->elem[i];
		bam1_t *mate = mFindBestMate(b, bamvector[1]);
		if (mate == NULL) {                    /* No mate */
			b->core.flag |= BAM_FMUNMAP;
			samwrite(output, b);
		} else {                               /* Mate found */

			int32_t  pos       = b->core.pos;
			int32_t  mpos      = mate->core.pos;

			/* set self/mate template ids to match */

			   b->core.mtid = mate->core.tid;
			mate->core.mtid = b->core.tid;

			/* set mate positions */

			   b->core.mpos = mpos;
			mate->core.mpos = pos;

			if (b->core.tid == b->core.mtid) {
				uint32_t flag      = b->core.flag;
				uint32_t mate_flag = mate->core.flag;
				int32_t  self_reverse = ((     flag&BAM_FREVERSE)==BAM_FREVERSE);
				int32_t  mate_reverse = ((mate_flag&BAM_FREVERSE)==BAM_FREVERSE);
				int32_t  self_first;

				/* set read order in template */

				if (pos < mpos) {
					flag      |= BAM_FREAD1;
					mate_flag |= BAM_FREAD2;
					self_first = 1;
				} else {
					flag      |= BAM_FREAD2;
					mate_flag |= BAM_FREAD1;
					self_first = 0;
				}

				/* set mate strand flags */

				if (self_reverse)
					mate_flag |= BAM_FMREVERSE;
				if (mate_reverse)
					flag |= BAM_FMREVERSE;

				/* categorize paired end status */

				if (self_reverse == mate_reverse)      {    /* same direction */
					flag      |= M_BAM_FUNIDIR;
					mate_flag |= M_BAM_FUNIDIR;
				} else if (self_reverse == self_first) {    /* outie */
					flag      |= M_BAM_FOUTIE;
					mate_flag |= M_BAM_FOUTIE;
				} else {                                    /* proper pairing */
					
					int32_t insert_size;

					/* these two are properly paired */

					flag      |= BAM_FPROPER_PAIR;
					mate_flag |= BAM_FPROPER_PAIR;

					/* if i am reverse, mate's mate is reversed */
					/* if i am reverse, then calculate my end position in the refgenome for insert size calculation */

					if (flag&BAM_FREVERSE) {
						pos = bam_calend(&b->core, bam1_cigar(b));
					}
					if (mate_flag&BAM_FREVERSE) {
						mpos = bam_calend(&mate->core, bam1_cigar(mate));
					}

					/* calculate insert size properly now */

					if (b->core.mtid == b->core.tid) {
						insert_size = abs(pos-mpos) + 1;
						if (pos < mpos) {
							b->core.isize = insert_size;
							mate->core.isize = -insert_size;
						} else {
							b->core.isize = -insert_size;
							mate->core.isize = insert_size;
						}
					}
				}

				/* reset flags */

				b->core.flag    = flag;
				mate->core.flag = mate_flag;
			}

			/* write self */

			samwrite(output, b);
		}
	}

	for (i=0; i<bamvector[1]->size; i++) {
		bam1_t *b = bamvector[1]->elem[i];
		samwrite(output, b);
	}
#endif /* MAUI_OLD */

	/* free memory */

	mEmptyBamVector(bamvector[0]);
	mEmptyBamVector(bamvector[1]);

	return 1;
}

void mAddQuality(samfile_t *input, samfile_t *output, char *qualfile) {

	/* quality related */

	FILE      *qualstream;
	mQual     *sub  = (mQual*) mCalloc(1, sizeof(mQual));
	mQual     *qual = (mQual*) mCalloc(1, sizeof(mQual));

	/* BAM/SAM related */

	bam1_t    *b;
	uint32_t  *cigar;

	/* open qual stream */

	if ((qualstream = fopen(qualfile, "r")) == NULL)
		mDie("Cannot open qual file %s for reading", qualfile);

	/* read first qual */

	mReadQual(qualstream, qual);

	/* parse sam file */

	b = bam_init1();
	while (samread(input, b) >= 0) {
		bam1_core_t  *c;
		int           lclip     = 0;
		int           rclip     = 0;
		char         *fastq_qual;
		int           frag_length;

		/* get the riqht quality entry */

		if (strcmp(bam1_qname(b), qual->def) != 0) { /* dont have the quality of this read, so get it now */
			for (;;) {
				int qstatus = mReadQual(qualstream, qual);
				if (strcmp(qual->def, bam1_qname(b)) == 0)
					break; /* found hit */
				mFreeQual(qual);
				/* trouble below! */
				if (qstatus == END_OF_QUAL)
					mDie("Sequence quality unavailable for read '%s'!\nAre you sure you are using the right quality file?\n", bam1_qname(b));
			}
		}

		/* now have the right quality entry */

		c = &b->core;
		cigar = bam1_cigar(b);

		/* calculate left and right hard clips */

		if ((cigar[0]&0xf) == BAM_CHARD_CLIP)
			lclip = cigar[0]>>4;
		if ((cigar[c->n_cigar-1]&0xf) == BAM_CHARD_CLIP)
			rclip = cigar[c->n_cigar-1]>>4;

		/* frag length after clippint */

		frag_length = qual->length - rclip - lclip;

		/* get the quality string as bytes */
		/* I add the fastq bytes and not string, since it is stored this way in the bam object. Saves time for me! */

		if (c->flag&BAM_FREVERSE) {
			sub = mSubQual(qual, rclip, frag_length);
			fastq_qual = mFastqBytes(sub, 1);
		} else {
			sub = mSubQual(qual, lclip, frag_length);
			fastq_qual = mFastqBytes(sub, 0);
		}

		/*printf("%128: %s\n", qual->def, fastq_qual);*/

		/* add quality to bam object */

		memcpy(bam1_qual(b), fastq_qual, frag_length);

		/* write bam object */

		samwrite(output, b);
		mFree(fastq_qual);
	}
	bam_destroy1(b);

	mFreeQual(qual);
	mFree(qual);
	mFree(sub);
}

void mWriteBestHitBamPool(samfile_t *stream, mBamPool *pool) {
	int i;
	bam1_t **elem = pool->elem;
	int *score = (int*) mCalloc(pool->limit, sizeof(int));
	int32_t best_score = INT32_MIN;
	
	if (pool->origin <= pool->current) {
		for (i=pool->origin; i<pool->current; i++)  {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
				}
			}
		}
		for (i=pool->origin; i<pool->current; i++)  {
			if (score[i] == best_score) {
				samwrite(stream, elem[i]);
			}
		}
	} else {
		for (i=pool->origin; i<pool->limit; i++) {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
				}
			}
		}
		for (i=0; i<pool->current; i++) {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
				}
			}
		}
		for (i=pool->origin; i<pool->limit; i++) {
			if (score[i] == best_score) {
				samwrite(stream, elem[i]);
			}
		}
		for (i=0; i<pool->current; i++) {
			if (score[i] == best_score) {
				samwrite(stream, elem[i]);
			}
		}
	}
	mFree(score);
}

void mWriteUniqueBestHitBamPool(samfile_t *stream, mBamPool *pool) {
	int        i;
	int        best_count = 0;
	int32_t    best_score = INT32_MIN;
	int       *score      = (int*) mCalloc(pool->limit, sizeof(int));
	bam1_t   **elem       = pool->elem;
	
	if (pool->origin <= pool->current) {

		/* track the best score and its count */

		for (i=pool->origin; i<pool->current; i++)  {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
					best_count = 1;
				} else if (score[i] == best_score) {
					best_count++;
				}
			}
		}

		/* output only if the best hit is uniq with count=1 */

		if (best_count == 1) {
			for (i=pool->origin; i<pool->current; i++)  {
				if (score[i] == best_score) {
					samwrite(stream, elem[i]);
				}
			}
		}
	} else {
		for (i=pool->origin; i<pool->limit; i++) {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
					best_count = 1;
				} else if (score[i] == best_score) {
					best_count++;
				}
			}
		}
		for (i=0; i<pool->current; i++) {
			uint8_t *as = bam_aux_get(elem[i], "AS");
			if (as) {
				score[i] = bam_aux2i(as);
				if (score[i] > best_score) {
					best_score = score[i];
					best_count = 1;
				} else if (score[i] == best_score) {
					best_count++;
				}
			}
		}
		if (best_count == 1) {
			for (i=pool->origin; i<pool->limit; i++) {
				if (score[i] == best_score) {
					samwrite(stream, elem[i]);
				}
			}
			for (i=0; i<pool->current; i++) {
				if (score[i] == best_score) {
					samwrite(stream, elem[i]);
				}
			}
		}
	}
	mFree(score);
}
